"""
HB_Delete V1.1

Last Modified: Oct/11/2018
Works with CINEMA 4D R16.050 and up.
Copyright: Holger Biebrach, www.c4dstuff.com

Name-US: HB_Delete
Description-US: Deletes or Dissolves Components or Objects [SHIFT: Delete Connected/Delete Children, CTRL: Dissolve/Delete without Children, CTRL + SHIFT: Delete Object]

This Script does following things:

In Components Mode (Polygon/Edge/Point):
- Deletes selected Components and removes unused Points (which remain when you usualy delete Polygons)
- SHIFT: Removes all connected components
- CTRL: Dissolves selected Components
- CTRL + SHIFT: Removes Object even when you are in Componentsmode

In Object Mode:
- Deletes Objects.
- SHIFT: Deletes all Children of selected Objects.
- CTRL: Deletes Current selected Objects without the Children.

Video Tutorial:
https://youtu.be/eEcAttV56kE?t=17m24s


Name-DE: HB_Delete
Description-DE: Wechselt zwischen Punkte/Kanten/Polygon Modus und macht ein paar andere praktische dinge. [SHIFT: Delete Connected/Delete Children, CTRL: Dissolve/Delete without Children, CTRL + SHIFT: Delete Object]

ChangeLog:
Jun/11/2015 V1.0
- Release Version

Oct/11/2018 V1.1
- New Highres Icon



"""

import c4d
from c4d import utils

def DeleteObjects(objList):
    for obj in objList:

        doc.AddUndo(c4d.UNDOTYPE_DELETE, obj)
        obj.Remove()
        c4d.EventAdd()

def DeleteChildren(obj):
    for obj in objList:

        children=obj.GetChildren()
        for child in children:

            doc.AddUndo(c4d.UNDOTYPE_DELETE,child )
            child.Remove()

    c4d.EventAdd()

def DeleteConnectedorChildren(objList):
    for obj in objList:

        if obj.GetType()==c4d.Opolygon and not c4d.IsCommandChecked(12298): #Object Polygonal and not Modelmode
                c4d.CallCommand(12557) # Select Connected

                c4d.CallCommand(12109) # Delete

                settings = c4d.BaseContainer()
                settings[c4d.MDATA_OPTIMIZE_UNUSEDPOINTS] = True
                settings[c4d.MDATA_OPTIMIZE_POINTS] = False
                settings[c4d.MDATA_OPTIMIZE_POLYGONS] = False
                utils.SendModelingCommand(command = c4d.MCOMMAND_OPTIMIZE, list = [obj], mode =c4d.MODELINGCOMMANDMODE_ALL, bc = settings, doc = doc)
        else:
            children=obj.GetChildren()
            for child in children:
                doc.AddUndo(c4d.UNDOTYPE_DELETE, child)
                child.Remove()

        c4d.EventAdd()

def DeleteParentWithoutChildren(objList):
    for obj in objList:
        if not c4d.IsCommandChecked(12298): # not Modelmode
            c4d.CallCommand(440000043) # Dissolve

        else: #Modelmode

            children=obj.GetChildren()
            for child in children:
                Pos=child.GetMg()
                doc.AddUndo(c4d.UNDOTYPE_CHANGE, child)
                child.InsertAfter(obj)
                doc.AddUndo(c4d.UNDOTYPE_CHANGE, child)
                child.SetMg(Pos)
                child.SetBit(c4d.BIT_ACTIVE)


            doc.AddUndo(c4d.UNDOTYPE_DELETE, obj)
            obj.Remove()


        c4d.EventAdd()


def main():

    doc.StartUndo()
    bc = c4d.BaseContainer()
    if c4d.gui.GetInputState(c4d.BFM_INPUT_KEYBOARD,c4d.BFM_INPUT_CHANNEL,bc):


        if bc[c4d.BFM_INPUT_QUALIFIER] ==1 :    # SHIFT Click
            DeleteConnectedorChildren(doc.GetSelection())


        if bc[c4d.BFM_INPUT_QUALIFIER] ==0 or  bc[c4d.BFM_INPUT_QUALIFIER] ==4 : # No Modifier or ALT

            selobj= doc.GetActiveObject()
            c4d.CallCommand(12109) # Delete

            if selobj.GetType()==c4d.Opolygon:
                settings = c4d.BaseContainer()
                settings[c4d.MDATA_OPTIMIZE_UNUSEDPOINTS] = True
                settings[c4d.MDATA_OPTIMIZE_POINTS] = False
                settings[c4d.MDATA_OPTIMIZE_POLYGONS] = False
                utils.SendModelingCommand(command = c4d.MCOMMAND_OPTIMIZE, list = [selobj], mode =c4d.MODELINGCOMMANDMODE_ALL, bc = settings, doc = doc)

        if bc[c4d.BFM_INPUT_QUALIFIER] ==2 : #CTRL-Click
            DeleteParentWithoutChildren(doc.GetSelection())

        if bc[c4d.BFM_INPUT_QUALIFIER] ==3 : # CTRL + SHIFT Click

            DeleteObjects(doc.GetSelection())

    doc.EndUndo()


if __name__=='__main__':
    main()